/*
 * Decompiled with CFR 0.152.
 */
package org.hornetq.core.journal.impl;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicLong;
import org.hornetq.api.core.HornetQBuffer;
import org.hornetq.api.core.HornetQBuffers;
import org.hornetq.api.core.HornetQInterruptedException;
import org.hornetq.core.journal.EncodingSupport;
import org.hornetq.core.journal.IOAsyncTask;
import org.hornetq.core.journal.impl.TimedBufferObserver;
import org.hornetq.core.journal.impl.dataformat.ByteArrayEncoding;
import org.hornetq.journal.HornetQJournalLogger;

public class TimedBuffer {
    public static final int MAX_CHECKS_ON_SLEEP = 20;
    private TimedBufferObserver bufferObserver;
    private final Semaphore spinLimiter = new Semaphore(1);
    private CheckTimer timerRunnable = new CheckTimer();
    private final int bufferSize;
    private final HornetQBuffer buffer;
    private int bufferLimit = 0;
    private List<IOAsyncTask> callbacks;
    private volatile int timeout;
    private volatile boolean pendingSync = false;
    private Thread timerThread;
    private volatile boolean started;
    private boolean delayFlush;
    private final boolean logRates;
    private final AtomicLong bytesFlushed = new AtomicLong(0L);
    private final AtomicLong flushesDone = new AtomicLong(0L);
    private Timer logRatesTimer;
    private TimerTask logRatesTimerTask;
    private boolean useSleep = true;
    private boolean spinning = false;

    public TimedBuffer(int size, int timeout, boolean logRates) {
        this.bufferSize = size;
        this.logRates = logRates;
        if (logRates) {
            this.logRatesTimer = new Timer(true);
        }
        this.buffer = HornetQBuffers.fixedBuffer((int)this.bufferSize);
        this.buffer.clear();
        this.bufferLimit = 0;
        this.callbacks = new ArrayList<IOAsyncTask>();
        this.timeout = timeout;
    }

    public synchronized boolean isUseSleep() {
        return this.useSleep;
    }

    public synchronized void setUseSleep(boolean useSleep) {
        this.useSleep = useSleep;
    }

    public synchronized void start() {
        if (this.started) {
            return;
        }
        try {
            this.spinLimiter.acquire();
        }
        catch (InterruptedException e) {
            throw new HornetQInterruptedException((Throwable)e);
        }
        this.timerRunnable = new CheckTimer();
        this.timerThread = new Thread((Runnable)this.timerRunnable, "hornetq-buffer-timeout");
        this.timerThread.start();
        if (this.logRates) {
            this.logRatesTimerTask = new LogRatesTimerTask();
            this.logRatesTimer.scheduleAtFixedRate(this.logRatesTimerTask, 2000L, 2000L);
        }
        this.started = true;
    }

    public void stop() {
        if (!this.started) {
            return;
        }
        this.flush();
        this.bufferObserver = null;
        this.timerRunnable.close();
        this.spinLimiter.release();
        if (this.logRates) {
            this.logRatesTimerTask.cancel();
        }
        while (this.timerThread.isAlive()) {
            try {
                this.timerThread.join();
            }
            catch (InterruptedException e) {
                throw new HornetQInterruptedException((Throwable)e);
            }
        }
        this.started = false;
    }

    public synchronized void setObserver(TimedBufferObserver observer) {
        if (this.bufferObserver != null) {
            this.flush();
        }
        this.bufferObserver = observer;
    }

    public synchronized boolean checkSize(int sizeChecked) {
        if (!this.started) {
            throw new IllegalStateException("TimedBuffer is not started");
        }
        if (sizeChecked > this.bufferSize) {
            throw new IllegalStateException("Can't write records bigger than the bufferSize(" + this.bufferSize + ") on the journal");
        }
        if (this.bufferLimit == 0 || this.buffer.writerIndex() + sizeChecked > this.bufferLimit) {
            this.flush();
            this.delayFlush = true;
            int remainingInFile = this.bufferObserver.getRemainingBytes();
            if (sizeChecked > remainingInFile) {
                return false;
            }
            this.bufferLimit = Math.min(remainingInFile, this.bufferSize);
            return true;
        }
        this.delayFlush = true;
        return true;
    }

    public synchronized void addBytes(HornetQBuffer bytes, boolean sync, IOAsyncTask callback) {
        this.addBytes(new ByteArrayEncoding(bytes.toByteBuffer().array()), sync, callback);
    }

    public synchronized void addBytes(EncodingSupport bytes, boolean sync, IOAsyncTask callback) {
        if (!this.started) {
            throw new IllegalStateException("TimedBuffer is not started");
        }
        this.delayFlush = false;
        bytes.encode(this.buffer);
        this.callbacks.add(callback);
        if (sync) {
            this.pendingSync = true;
            this.startSpin();
        }
    }

    public void flush() {
        this.flush(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flush(boolean force) {
        TimedBuffer timedBuffer = this;
        synchronized (timedBuffer) {
            if (!this.started) {
                throw new IllegalStateException("TimedBuffer is not started");
            }
            if ((force || !this.delayFlush) && this.buffer.writerIndex() > 0) {
                int pos = this.buffer.writerIndex();
                if (this.logRates) {
                    this.bytesFlushed.addAndGet(pos);
                }
                ByteBuffer bufferToFlush = this.bufferObserver.newBuffer(this.bufferSize, pos);
                bufferToFlush.put(this.buffer.toByteBuffer().array(), 0, pos);
                this.bufferObserver.flushBuffer(bufferToFlush, this.pendingSync, this.callbacks);
                this.stopSpin();
                this.pendingSync = false;
                this.callbacks = new LinkedList<IOAsyncTask>();
                this.buffer.clear();
                this.bufferLimit = 0;
                this.flushesDone.incrementAndGet();
            }
        }
    }

    protected void sleep(int sleepMillis, int sleepNanos) throws InterruptedException {
        Thread.sleep(sleepMillis, sleepNanos);
    }

    protected void stopSpin() {
        if (this.spinning) {
            try {
                this.spinLimiter.acquire();
            }
            catch (InterruptedException e) {
                throw new HornetQInterruptedException((Throwable)e);
            }
            this.spinning = false;
        }
    }

    protected void startSpin() {
        if (!this.spinning) {
            this.spinLimiter.release();
            this.spinning = true;
        }
    }

    private class CheckTimer
    implements Runnable {
        private volatile boolean closed = false;
        int checks = 0;
        int failedChecks = 0;
        long timeBefore = 0L;
        final int sleepMillis = TimedBuffer.access$400(TimedBuffer.this) / 1000000;
        final int sleepNanos = TimedBuffer.access$400(TimedBuffer.this) % 1000000;

        private CheckTimer() {
        }

        @Override
        public void run() {
            long lastFlushTime = 0L;
            while (!this.closed) {
                if (TimedBuffer.this.pendingSync) {
                    if (TimedBuffer.this.isUseSleep()) {
                        TimedBuffer.this.flush();
                        lastFlushTime = System.nanoTime();
                    } else if (TimedBuffer.this.bufferObserver != null && System.nanoTime() > lastFlushTime + (long)TimedBuffer.this.timeout) {
                        TimedBuffer.this.flush();
                        lastFlushTime = System.nanoTime();
                    }
                }
                this.sleepIfPossible();
                try {
                    TimedBuffer.this.spinLimiter.acquire();
                    Thread.yield();
                    TimedBuffer.this.spinLimiter.release();
                }
                catch (InterruptedException e) {
                    throw new HornetQInterruptedException((Throwable)e);
                }
            }
        }

        private void sleepIfPossible() {
            if (TimedBuffer.this.isUseSleep()) {
                if (this.checks < 20) {
                    this.timeBefore = System.nanoTime();
                }
                try {
                    TimedBuffer.this.sleep(this.sleepMillis, this.sleepNanos);
                }
                catch (InterruptedException e) {
                    throw new HornetQInterruptedException((Throwable)e);
                }
                catch (Exception e) {
                    TimedBuffer.this.setUseSleep(false);
                    HornetQJournalLogger.LOGGER.warn(e.getMessage() + ", disabling sleep on TimedBuffer, using spin now", e);
                }
                if (this.checks < 20) {
                    long realTimeSleep = System.nanoTime() - this.timeBefore;
                    if ((double)realTimeSleep > (double)TimedBuffer.this.timeout * 1.5) {
                        ++this.failedChecks;
                    }
                    if (++this.checks >= 20 && (double)this.failedChecks > 10.0) {
                        HornetQJournalLogger.LOGGER.debug("Thread.sleep with nano seconds is not working as expected, Your kernel possibly doesn't support real time. the Journal TimedBuffer will spin for timeouts");
                        TimedBuffer.this.setUseSleep(false);
                    }
                }
            }
        }

        public void close() {
            this.closed = true;
        }
    }

    private class LogRatesTimerTask
    extends TimerTask {
        private boolean closed;
        private long lastExecution;
        private long lastBytesFlushed;
        private long lastFlushesDone;

        private LogRatesTimerTask() {
        }

        @Override
        public synchronized void run() {
            if (!this.closed) {
                long now = System.currentTimeMillis();
                long bytesF = TimedBuffer.this.bytesFlushed.get();
                long flushesD = TimedBuffer.this.flushesDone.get();
                if (this.lastExecution != 0L) {
                    double rate = 1000.0 * (double)(bytesF - this.lastBytesFlushed) / (double)(now - this.lastExecution);
                    HornetQJournalLogger.LOGGER.writeRate(rate, (long)(rate / 1048576.0));
                    double flushRate = 1000.0 * (double)(flushesD - this.lastFlushesDone) / (double)(now - this.lastExecution);
                    HornetQJournalLogger.LOGGER.flushRate(flushRate);
                }
                this.lastExecution = now;
                this.lastBytesFlushed = bytesF;
                this.lastFlushesDone = flushesD;
            }
        }

        @Override
        public synchronized boolean cancel() {
            this.closed = true;
            return super.cancel();
        }
    }
}

