/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.analysis.interpolation;

import java.io.Serializable;
import java.util.Arrays;
import org.apache.commons.math.MathException;
import org.apache.commons.math.analysis.interpolation.SplineInterpolator;
import org.apache.commons.math.analysis.interpolation.UnivariateRealInterpolator;
import org.apache.commons.math.analysis.polynomials.PolynomialSplineFunction;

public class LoessInterpolator
implements UnivariateRealInterpolator,
Serializable {
    private static final long serialVersionUID = 5204927143605193821L;
    public static final double DEFAULT_BANDWIDTH = 0.3;
    public static final int DEFAULT_ROBUSTNESS_ITERS = 2;
    private final double bandwidth;
    private final int robustnessIters;

    public LoessInterpolator() {
        this.bandwidth = 0.3;
        this.robustnessIters = 2;
    }

    public LoessInterpolator(double bandwidth, int robustnessIters) throws MathException {
        if (bandwidth < 0.0 || bandwidth > 1.0) {
            throw new MathException("bandwidth must be in the interval [0,1], but got {0}", bandwidth);
        }
        this.bandwidth = bandwidth;
        if (robustnessIters < 0) {
            throw new MathException("the number of robustness iterations must be non-negative, but got {0}", robustnessIters);
        }
        this.robustnessIters = robustnessIters;
    }

    public final PolynomialSplineFunction interpolate(double[] xval, double[] yval) throws MathException {
        return new SplineInterpolator().interpolate(xval, this.smooth(xval, yval));
    }

    public final double[] smooth(double[] xval, double[] yval) throws MathException {
        if (xval.length != yval.length) {
            throw new MathException("Loess expects the abscissa and ordinate arrays to be of the same size, but got {0} abscisssae and {1} ordinatae", xval.length, yval.length);
        }
        int n = xval.length;
        if (n == 0) {
            throw new MathException("Loess expects at least 1 point", new Object[0]);
        }
        LoessInterpolator.checkAllFiniteReal(xval, true);
        LoessInterpolator.checkAllFiniteReal(yval, false);
        LoessInterpolator.checkStrictlyIncreasing(xval);
        if (n == 1) {
            return new double[]{yval[0]};
        }
        if (n == 2) {
            return new double[]{yval[0], yval[1]};
        }
        int bandwidthInPoints = (int)(this.bandwidth * (double)n);
        if (bandwidthInPoints < 2) {
            throw new MathException("the bandwidth must be large enough to accomodate at least 2 points. There are {0}  data points, and bandwidth must be at least {1}  but it is only {2}", n, 2.0 / (double)n, this.bandwidth);
        }
        double[] res = new double[n];
        double[] residuals = new double[n];
        double[] sortedResiduals = new double[n];
        double[] robustnessWeights = new double[n];
        Arrays.fill(robustnessWeights, 1.0);
        for (int iter = 0; iter <= this.robustnessIters; ++iter) {
            int[] bandwidthInterval = new int[]{0, bandwidthInPoints - 1};
            for (int i = 0; i < n; ++i) {
                int iright;
                int ileft;
                double x = xval[i];
                if (i > 0) {
                    LoessInterpolator.updateBandwidthInterval(xval, i, bandwidthInterval);
                }
                int edge = xval[i] - xval[ileft = bandwidthInterval[0]] > xval[iright = bandwidthInterval[1]] - xval[i] ? ileft : iright;
                double sumWeights = 0.0;
                double sumX = 0.0;
                double sumXSquared = 0.0;
                double sumY = 0.0;
                double sumXY = 0.0;
                double denom = Math.abs(1.0 / (xval[edge] - x));
                for (int k = ileft; k <= iright; ++k) {
                    double xk = xval[k];
                    double yk = yval[k];
                    double dist = k < i ? x - xk : xk - x;
                    double w = LoessInterpolator.tricube(dist * denom) * robustnessWeights[k];
                    double xkw = xk * w;
                    sumWeights += w;
                    sumX += xkw;
                    sumXSquared += xk * xkw;
                    sumY += yk * w;
                    sumXY += yk * xkw;
                }
                double meanX = sumX / sumWeights;
                double meanY = sumY / sumWeights;
                double meanXY = sumXY / sumWeights;
                double meanXSquared = sumXSquared / sumWeights;
                double beta = meanXSquared == meanX * meanX ? 0.0 : (meanXY - meanX * meanY) / (meanXSquared - meanX * meanX);
                double alpha = meanY - beta * meanX;
                res[i] = beta * x + alpha;
                residuals[i] = Math.abs(yval[i] - res[i]);
            }
            if (iter == this.robustnessIters) break;
            System.arraycopy(residuals, 0, sortedResiduals, 0, n);
            Arrays.sort(sortedResiduals);
            double medianResidual = sortedResiduals[n / 2];
            if (medianResidual == 0.0) break;
            for (int i = 0; i < n; ++i) {
                double arg = residuals[i] / (6.0 * medianResidual);
                robustnessWeights[i] = arg >= 1.0 ? 0.0 : Math.pow(1.0 - arg * arg, 2.0);
            }
        }
        return res;
    }

    private static void updateBandwidthInterval(double[] xval, int i, int[] bandwidthInterval) {
        int left = bandwidthInterval[0];
        int right = bandwidthInterval[1];
        if (right < xval.length - 1 && xval[right + 1] - xval[i] < xval[i] - xval[left]) {
            bandwidthInterval[0] = bandwidthInterval[0] + 1;
            bandwidthInterval[1] = bandwidthInterval[1] + 1;
        }
    }

    private static double tricube(double x) {
        double tmp = 1.0 - x * x * x;
        return tmp * tmp * tmp;
    }

    private static void checkAllFiniteReal(double[] values, boolean isAbscissae) throws MathException {
        for (int i = 0; i < values.length; ++i) {
            double x = values[i];
            if (!Double.isInfinite(x) && !Double.isNaN(x)) continue;
            String pattern = isAbscissae ? "all abscissae must be finite real numbers, but {0}-th is {1}" : "all ordinatae must be finite real numbers, but {0}-th is {1}";
            throw new MathException(pattern, i, x);
        }
    }

    private static void checkStrictlyIncreasing(double[] xval) throws MathException {
        for (int i = 0; i < xval.length; ++i) {
            if (i < 1 || !(xval[i - 1] >= xval[i])) continue;
            throw new MathException("the abscissae array must be sorted in a strictly increasing order, but the {0}-th element is {1} whereas {2}-th is {3}", i - 1, xval[i - 1], i, xval[i]);
        }
    }
}

