/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.linear;

import org.apache.commons.math.MathRuntimeException;
import org.apache.commons.math.linear.Array2DRowRealMatrix;
import org.apache.commons.math.linear.BiDiagonalTransformer;
import org.apache.commons.math.linear.DecompositionSolver;
import org.apache.commons.math.linear.DefaultRealMatrixPreservingVisitor;
import org.apache.commons.math.linear.EigenDecomposition;
import org.apache.commons.math.linear.EigenDecompositionImpl;
import org.apache.commons.math.linear.InvalidMatrixException;
import org.apache.commons.math.linear.MatrixUtils;
import org.apache.commons.math.linear.RealMatrix;
import org.apache.commons.math.linear.RealVector;
import org.apache.commons.math.linear.SingularMatrixException;
import org.apache.commons.math.linear.SingularValueDecomposition;

public class SingularValueDecompositionImpl
implements SingularValueDecomposition {
    private int m;
    private int n;
    private BiDiagonalTransformer transformer;
    private double[] mainBidiagonal;
    private double[] secondaryBidiagonal;
    private double[] mainTridiagonal;
    private double[] secondaryTridiagonal;
    private EigenDecomposition eigenDecomposition;
    private double[] singularValues;
    private RealMatrix cachedU;
    private RealMatrix cachedUt;
    private RealMatrix cachedS;
    private RealMatrix cachedV;
    private RealMatrix cachedVt;

    public SingularValueDecompositionImpl(RealMatrix matrix) throws InvalidMatrixException {
        int i;
        this.m = matrix.getRowDimension();
        this.n = matrix.getColumnDimension();
        this.cachedU = null;
        this.cachedS = null;
        this.cachedV = null;
        this.cachedVt = null;
        this.transformer = new BiDiagonalTransformer(matrix);
        this.mainBidiagonal = this.transformer.getMainDiagonalRef();
        this.secondaryBidiagonal = this.transformer.getSecondaryDiagonalRef();
        this.mainTridiagonal = new double[this.mainBidiagonal.length];
        this.secondaryTridiagonal = new double[this.mainBidiagonal.length - 1];
        double a = this.mainBidiagonal[0];
        this.mainTridiagonal[0] = a * a;
        for (i = 1; i < this.mainBidiagonal.length; ++i) {
            double b = this.secondaryBidiagonal[i - 1];
            this.secondaryTridiagonal[i - 1] = a * b;
            a = this.mainBidiagonal[i];
            this.mainTridiagonal[i] = a * a + b * b;
        }
        this.eigenDecomposition = new EigenDecompositionImpl(this.mainTridiagonal, this.secondaryTridiagonal, Double.MIN_NORMAL);
        this.singularValues = this.eigenDecomposition.getRealEigenvalues();
        for (i = 0; i < this.singularValues.length; ++i) {
            this.singularValues[i] = Math.sqrt(this.singularValues[i]);
        }
    }

    public RealMatrix getU() throws InvalidMatrixException {
        if (this.cachedU == null) {
            if (this.m >= this.n) {
                int j;
                double[][] eData = this.eigenDecomposition.getV().getData();
                double[][] iData = new double[this.m][];
                double[] ei1 = eData[0];
                iData[0] = ei1;
                for (int i = 0; i < this.n - 1; ++i) {
                    double[] ei0 = ei1;
                    ei1 = eData[i + 1];
                    iData[i + 1] = ei1;
                    for (j = 0; j < this.n; ++j) {
                        ei0[j] = (this.mainBidiagonal[i] * ei0[j] + this.secondaryBidiagonal[i] * ei1[j]) / this.singularValues[j];
                    }
                }
                double lastMain = this.mainBidiagonal[this.n - 1];
                for (j = 0; j < this.n; ++j) {
                    int n = j;
                    ei1[n] = ei1[n] * (lastMain / this.singularValues[j]);
                }
                for (int i = this.n; i < this.m; ++i) {
                    iData[i] = new double[this.n];
                }
                this.cachedU = this.transformer.getU().multiply(MatrixUtils.createRealMatrix(iData));
            } else {
                this.cachedU = this.transformer.getU().multiply(this.eigenDecomposition.getV());
            }
        }
        return this.cachedU;
    }

    public RealMatrix getUT() throws InvalidMatrixException {
        if (this.cachedUt == null) {
            this.cachedUt = this.getU().transpose();
        }
        return this.cachedUt;
    }

    public RealMatrix getS() throws InvalidMatrixException {
        if (this.cachedS == null) {
            this.cachedS = MatrixUtils.createRealDiagonalMatrix(this.singularValues);
        }
        return this.cachedS;
    }

    public double[] getSingularValues() throws InvalidMatrixException {
        return (double[])this.singularValues.clone();
    }

    public RealMatrix getV() throws InvalidMatrixException {
        if (this.cachedV == null) {
            if (this.m >= this.n) {
                this.cachedV = this.transformer.getV().multiply(this.eigenDecomposition.getV());
            } else {
                int j;
                double[][] eData = this.eigenDecomposition.getV().getData();
                double[][] iData = new double[this.n][];
                double[] ei1 = eData[0];
                iData[0] = ei1;
                for (int i = 0; i < this.m - 1; ++i) {
                    double[] ei0 = ei1;
                    ei1 = eData[i + 1];
                    iData[i + 1] = ei1;
                    for (j = 0; j < this.m; ++j) {
                        ei0[j] = (this.mainBidiagonal[i] * ei0[j] + this.secondaryBidiagonal[i] * ei1[j]) / this.singularValues[j];
                    }
                }
                double lastMain = this.mainBidiagonal[this.m - 1];
                for (j = 0; j < this.m; ++j) {
                    int n = j;
                    ei1[n] = ei1[n] * (lastMain / this.singularValues[j]);
                }
                for (int i = this.m; i < this.n; ++i) {
                    iData[i] = new double[this.m];
                }
                this.cachedV = this.transformer.getV().multiply(MatrixUtils.createRealMatrix(iData));
            }
        }
        return this.cachedV;
    }

    public RealMatrix getVT() throws InvalidMatrixException {
        if (this.cachedVt == null) {
            this.cachedVt = this.getV().transpose();
        }
        return this.cachedVt;
    }

    public RealMatrix getCovariance(double minSingularValue) {
        int dimension;
        for (dimension = 0; dimension < this.n && this.singularValues[dimension] >= minSingularValue; ++dimension) {
        }
        if (dimension == 0) {
            throw MathRuntimeException.createIllegalArgumentException("cutoff singular value is {0}, should be at most {1}", minSingularValue, this.singularValues[0]);
        }
        final double[][] data = new double[dimension][this.n];
        this.getVT().walkInOptimizedOrder(new DefaultRealMatrixPreservingVisitor(){

            public void visit(int row, int column, double value) {
                data[row][column] = value / SingularValueDecompositionImpl.this.singularValues[row];
            }
        }, 0, dimension - 1, 0, this.n - 1);
        Array2DRowRealMatrix jv = new Array2DRowRealMatrix(data, false);
        return jv.transpose().multiply(jv);
    }

    public double getNorm() throws InvalidMatrixException {
        return this.singularValues[0];
    }

    public double getConditionNumber() throws InvalidMatrixException {
        return this.singularValues[0] / this.singularValues[this.singularValues.length - 1];
    }

    public int getRank() throws IllegalStateException {
        double threshold = (double)Math.max(this.m, this.n) * Math.ulp(this.singularValues[0]);
        for (int i = this.singularValues.length - 1; i >= 0; --i) {
            if (!(this.singularValues[i] > threshold)) continue;
            return i + 1;
        }
        return 0;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.singularValues, this.getUT(), this.getV(), this.getRank() == this.singularValues.length);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[] singularValues;
        private final RealMatrix uT;
        private final RealMatrix v;
        private boolean nonSingular;

        private Solver(double[] singularValues, RealMatrix uT, RealMatrix v, boolean nonSingular) {
            this.singularValues = singularValues;
            this.uT = uT;
            this.v = v;
            this.nonSingular = nonSingular;
        }

        public double[] solve(double[] b) throws IllegalArgumentException, InvalidMatrixException {
            if (b.length != this.singularValues.length) {
                throw MathRuntimeException.createIllegalArgumentException("vector length mismatch: got {0} but expected {1}", b.length, this.singularValues.length);
            }
            double[] w = this.uT.operate(b);
            int i = 0;
            while (i < this.singularValues.length) {
                double si = this.singularValues[i];
                if (si == 0.0) {
                    throw new SingularMatrixException();
                }
                int n = i++;
                w[n] = w[n] / si;
            }
            return this.v.operate(w);
        }

        public RealVector solve(RealVector b) throws IllegalArgumentException, InvalidMatrixException {
            if (b.getDimension() != this.singularValues.length) {
                throw MathRuntimeException.createIllegalArgumentException("vector length mismatch: got {0} but expected {1}", b.getDimension(), this.singularValues.length);
            }
            RealVector w = this.uT.operate(b);
            for (int i = 0; i < this.singularValues.length; ++i) {
                double si = this.singularValues[i];
                if (si == 0.0) {
                    throw new SingularMatrixException();
                }
                w.setEntry(i, w.getEntry(i) / si);
            }
            return this.v.operate(w);
        }

        public RealMatrix solve(RealMatrix b) throws IllegalArgumentException, InvalidMatrixException {
            if (b.getRowDimension() != this.singularValues.length) {
                throw MathRuntimeException.createIllegalArgumentException("dimensions mismatch: got {0}x{1} but expected {2}x{3}", b.getRowDimension(), b.getColumnDimension(), this.singularValues.length, "n");
            }
            RealMatrix w = this.uT.multiply(b);
            for (int i = 0; i < this.singularValues.length; ++i) {
                double si = this.singularValues[i];
                if (si == 0.0) {
                    throw new SingularMatrixException();
                }
                double inv = 1.0 / si;
                for (int j = 0; j < b.getColumnDimension(); ++j) {
                    w.multiplyEntry(i, j, inv);
                }
            }
            return this.v.multiply(w);
        }

        public boolean isNonSingular() {
            return this.nonSingular;
        }

        public RealMatrix getInverse() throws InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.singularValues.length));
        }
    }
}

